/*:
 * @target MZ
 * @plugindesc v1.0 言語ブリッジ：Configと変数を安全に同期し、ゲーム中の切替を確実化
 * @author HS
 *
 * @param languageVarId
 * @text 言語インデックス変数ID
 * @type variable
 * @default 0
 * @desc HS_LangSimpleで使っている言語変数ID（例: 9）。未設定ならHS_LangSimple側の値を自動取得。
 *
 * @param autoSyncOnLoad
 * @text ロード成功時に Config→変数 同期
 * @type boolean
 * @default true
 *
 * @param autoSyncOnMapStart
 * @text マップ開始時も同期（通常OFF推奨）
 * @type boolean
 * @default false
 * @desc ONにするとマップ突入ごとにConfigの値で上書きします。ゲーム中切替を優先したいならOFFのまま。
 *
 * @command SetLanguage
 * @text 言語を設定（index）
 * @arg index
 * @type number
 * @min 0
 * @desc 言語番号を設定（Configと変数の両方を更新）
 *
 * @command SyncConfigToVar
 * @text 同期：Config → 変数
 *
 * @command SyncVarToConfig
 * @text 同期：変数 → Config
 */
(() => {
  const PLUGIN_NAME = "HS_LangBridge";
  const prm = PluginManager.parameters(PLUGIN_NAME);

  // 変数ID（未指定ならHS_LangSimpleの設定を拾う）
  function resolveVarId() {
    let id = Number(prm.languageVarId || 0);
    if (!id) {
      const p2 = PluginManager.parameters("HS_LangSimple");
      id = Number(p2.languageVarId || 0);
    }
    return id;
  }
  const VAR_ID  = resolveVarId();
  const AUTO_LD = prm.autoSyncOnLoad === "true";
  const AUTO_MP = prm.autoSyncOnMapStart === "true";

  function readCfg() {
    return Number(ConfigManager.hs_langIndex ?? 0);
  }
  function writeCfg(idx) {
    ConfigManager.hs_langIndex = Number(idx);
    if (ConfigManager.save) ConfigManager.save();
  }
  function writeVar(idx) {
    if (VAR_ID > 0 && window.$gameVariables) {
      $gameVariables.setValue(VAR_ID, Number(idx));
    }
  }
  function syncCfg2Var() { writeVar(readCfg()); }
  function syncVar2Cfg() {
    if (VAR_ID > 0 && window.$gameVariables) {
      writeCfg($gameVariables.value(VAR_ID));
    }
  }
  function setLang(idx) { writeCfg(idx); writeVar(idx); }

  PluginManager.registerCommand(PLUGIN_NAME, "SetLanguage",
    args => setLang(Number(args.index)));
  PluginManager.registerCommand(PLUGIN_NAME, "SyncConfigToVar", () => syncCfg2Var());
  PluginManager.registerCommand(PLUGIN_NAME, "SyncVarToConfig", () => syncVar2Cfg());

  // 競合しにくいSceneフックのみ使用
  if (AUTO_LD) {
    const _onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
    Scene_Load.prototype.onLoadSuccess = function() {
      _onLoadSuccess.call(this);
      syncCfg2Var();
    };
  }
  if (AUTO_MP) {
    const _mapStart = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
      _mapStart.call(this);
      syncCfg2Var();
    };
  }

  // 便利関数（スクリプトコマンドから使えます）
  window.HS_SetLanguage       = setLang;
  window.HS_SyncConfigToVar   = syncCfg2Var;
  window.HS_SyncVarToConfig   = syncVar2Cfg;
})();
